﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesisvideo/KinesisVideoRequest.h>
#include <aws/kinesisvideo/KinesisVideo_EXPORTS.h>

#include <utility>

namespace Aws {
namespace KinesisVideo {
namespace Model {

/**
 */
class UpdateStreamRequest : public KinesisVideoRequest {
 public:
  AWS_KINESISVIDEO_API UpdateStreamRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateStream"; }

  AWS_KINESISVIDEO_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the stream whose metadata you want to update.</p> <p>The stream
   * name is an identifier for the stream, and must be unique for each account and
   * region.</p>
   */
  inline const Aws::String& GetStreamName() const { return m_streamName; }
  inline bool StreamNameHasBeenSet() const { return m_streamNameHasBeenSet; }
  template <typename StreamNameT = Aws::String>
  void SetStreamName(StreamNameT&& value) {
    m_streamNameHasBeenSet = true;
    m_streamName = std::forward<StreamNameT>(value);
  }
  template <typename StreamNameT = Aws::String>
  UpdateStreamRequest& WithStreamName(StreamNameT&& value) {
    SetStreamName(std::forward<StreamNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the stream whose metadata you want to update.</p>
   */
  inline const Aws::String& GetStreamARN() const { return m_streamARN; }
  inline bool StreamARNHasBeenSet() const { return m_streamARNHasBeenSet; }
  template <typename StreamARNT = Aws::String>
  void SetStreamARN(StreamARNT&& value) {
    m_streamARNHasBeenSet = true;
    m_streamARN = std::forward<StreamARNT>(value);
  }
  template <typename StreamARNT = Aws::String>
  UpdateStreamRequest& WithStreamARN(StreamARNT&& value) {
    SetStreamARN(std::forward<StreamARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the stream whose metadata you want to update.</p>
   */
  inline const Aws::String& GetCurrentVersion() const { return m_currentVersion; }
  inline bool CurrentVersionHasBeenSet() const { return m_currentVersionHasBeenSet; }
  template <typename CurrentVersionT = Aws::String>
  void SetCurrentVersion(CurrentVersionT&& value) {
    m_currentVersionHasBeenSet = true;
    m_currentVersion = std::forward<CurrentVersionT>(value);
  }
  template <typename CurrentVersionT = Aws::String>
  UpdateStreamRequest& WithCurrentVersion(CurrentVersionT&& value) {
    SetCurrentVersion(std::forward<CurrentVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the device that is writing to the stream. </p>  <p> In the
   * current implementation, Kinesis Video Streams does not use this name. </p>
   *
   */
  inline const Aws::String& GetDeviceName() const { return m_deviceName; }
  inline bool DeviceNameHasBeenSet() const { return m_deviceNameHasBeenSet; }
  template <typename DeviceNameT = Aws::String>
  void SetDeviceName(DeviceNameT&& value) {
    m_deviceNameHasBeenSet = true;
    m_deviceName = std::forward<DeviceNameT>(value);
  }
  template <typename DeviceNameT = Aws::String>
  UpdateStreamRequest& WithDeviceName(DeviceNameT&& value) {
    SetDeviceName(std::forward<DeviceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The stream's media type. Use <code>MediaType</code> to specify the type of
   * content that the stream contains to the consumers of the stream. For more
   * information about media types, see <a
   * href="http://www.iana.org/assignments/media-types/media-types.xhtml">Media
   * Types</a>. If you choose to specify the <code>MediaType</code>, see <a
   * href="https://tools.ietf.org/html/rfc6838#section-4.2">Naming
   * Requirements</a>.</p> <p>To play video on the console, you must specify the
   * correct video type. For example, if the video in the stream is H.264, specify
   * <code>video/h264</code> as the <code>MediaType</code>.</p>
   */
  inline const Aws::String& GetMediaType() const { return m_mediaType; }
  inline bool MediaTypeHasBeenSet() const { return m_mediaTypeHasBeenSet; }
  template <typename MediaTypeT = Aws::String>
  void SetMediaType(MediaTypeT&& value) {
    m_mediaTypeHasBeenSet = true;
    m_mediaType = std::forward<MediaTypeT>(value);
  }
  template <typename MediaTypeT = Aws::String>
  UpdateStreamRequest& WithMediaType(MediaTypeT&& value) {
    SetMediaType(std::forward<MediaTypeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_streamName;
  bool m_streamNameHasBeenSet = false;

  Aws::String m_streamARN;
  bool m_streamARNHasBeenSet = false;

  Aws::String m_currentVersion;
  bool m_currentVersionHasBeenSet = false;

  Aws::String m_deviceName;
  bool m_deviceNameHasBeenSet = false;

  Aws::String m_mediaType;
  bool m_mediaTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisVideo
}  // namespace Aws
