﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ivschat/IvschatRequest.h>
#include <aws/ivschat/Ivschat_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ivschat {
namespace Model {

/**
 */
class DisconnectUserRequest : public IvschatRequest {
 public:
  AWS_IVSCHAT_API DisconnectUserRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DisconnectUser"; }

  AWS_IVSCHAT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Identifier of the room from which the user's clients should be disconnected.
   * Currently this must be an ARN.</p>
   */
  inline const Aws::String& GetRoomIdentifier() const { return m_roomIdentifier; }
  inline bool RoomIdentifierHasBeenSet() const { return m_roomIdentifierHasBeenSet; }
  template <typename RoomIdentifierT = Aws::String>
  void SetRoomIdentifier(RoomIdentifierT&& value) {
    m_roomIdentifierHasBeenSet = true;
    m_roomIdentifier = std::forward<RoomIdentifierT>(value);
  }
  template <typename RoomIdentifierT = Aws::String>
  DisconnectUserRequest& WithRoomIdentifier(RoomIdentifierT&& value) {
    SetRoomIdentifier(std::forward<RoomIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>ID of the user (connection) to disconnect from the room.</p>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  DisconnectUserRequest& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Reason for disconnecting the user.</p>
   */
  inline const Aws::String& GetReason() const { return m_reason; }
  inline bool ReasonHasBeenSet() const { return m_reasonHasBeenSet; }
  template <typename ReasonT = Aws::String>
  void SetReason(ReasonT&& value) {
    m_reasonHasBeenSet = true;
    m_reason = std::forward<ReasonT>(value);
  }
  template <typename ReasonT = Aws::String>
  DisconnectUserRequest& WithReason(ReasonT&& value) {
    SetReason(std::forward<ReasonT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_roomIdentifier;
  bool m_roomIdentifierHasBeenSet = false;

  Aws::String m_userId;
  bool m_userIdHasBeenSet = false;

  Aws::String m_reason;
  bool m_reasonHasBeenSet = false;
};

}  // namespace Model
}  // namespace ivschat
}  // namespace Aws
