﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/storagegateway/StorageGatewayRequest.h>
#include <aws/storagegateway/StorageGateway_EXPORTS.h>
#include <aws/storagegateway/model/CacheAttributes.h>
#include <aws/storagegateway/model/EndpointNetworkConfiguration.h>
#include <aws/storagegateway/model/Tag.h>

#include <utility>

namespace Aws {
namespace StorageGateway {
namespace Model {

/**
 */
class AssociateFileSystemRequest : public StorageGatewayRequest {
 public:
  AWS_STORAGEGATEWAY_API AssociateFileSystemRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssociateFileSystem"; }

  AWS_STORAGEGATEWAY_API Aws::String SerializePayload() const override;

  AWS_STORAGEGATEWAY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The user name of the user credential that has permission to access the root
   * share D$ of the Amazon FSx file system. The user account must belong to the
   * Amazon FSx delegated admin user group.</p>
   */
  inline const Aws::String& GetUserName() const { return m_userName; }
  inline bool UserNameHasBeenSet() const { return m_userNameHasBeenSet; }
  template <typename UserNameT = Aws::String>
  void SetUserName(UserNameT&& value) {
    m_userNameHasBeenSet = true;
    m_userName = std::forward<UserNameT>(value);
  }
  template <typename UserNameT = Aws::String>
  AssociateFileSystemRequest& WithUserName(UserNameT&& value) {
    SetUserName(std::forward<UserNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The password of the user credential.</p>
   */
  inline const Aws::String& GetPassword() const { return m_password; }
  inline bool PasswordHasBeenSet() const { return m_passwordHasBeenSet; }
  template <typename PasswordT = Aws::String>
  void SetPassword(PasswordT&& value) {
    m_passwordHasBeenSet = true;
    m_password = std::forward<PasswordT>(value);
  }
  template <typename PasswordT = Aws::String>
  AssociateFileSystemRequest& WithPassword(PasswordT&& value) {
    SetPassword(std::forward<PasswordT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique string value that you supply that is used by the FSx File Gateway to
   * ensure idempotent file system association creation.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  AssociateFileSystemRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetGatewayARN() const { return m_gatewayARN; }
  inline bool GatewayARNHasBeenSet() const { return m_gatewayARNHasBeenSet; }
  template <typename GatewayARNT = Aws::String>
  void SetGatewayARN(GatewayARNT&& value) {
    m_gatewayARNHasBeenSet = true;
    m_gatewayARN = std::forward<GatewayARNT>(value);
  }
  template <typename GatewayARNT = Aws::String>
  AssociateFileSystemRequest& WithGatewayARN(GatewayARNT&& value) {
    SetGatewayARN(std::forward<GatewayARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Amazon FSx file system to associate
   * with the FSx File Gateway.</p>
   */
  inline const Aws::String& GetLocationARN() const { return m_locationARN; }
  inline bool LocationARNHasBeenSet() const { return m_locationARNHasBeenSet; }
  template <typename LocationARNT = Aws::String>
  void SetLocationARN(LocationARNT&& value) {
    m_locationARNHasBeenSet = true;
    m_locationARN = std::forward<LocationARNT>(value);
  }
  template <typename LocationARNT = Aws::String>
  AssociateFileSystemRequest& WithLocationARN(LocationARNT&& value) {
    SetLocationARN(std::forward<LocationARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of up to 50 tags that can be assigned to the file system association.
   * Each tag is a key-value pair.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  AssociateFileSystemRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  AssociateFileSystemRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the storage used for the audit logs.</p>
   */
  inline const Aws::String& GetAuditDestinationARN() const { return m_auditDestinationARN; }
  inline bool AuditDestinationARNHasBeenSet() const { return m_auditDestinationARNHasBeenSet; }
  template <typename AuditDestinationARNT = Aws::String>
  void SetAuditDestinationARN(AuditDestinationARNT&& value) {
    m_auditDestinationARNHasBeenSet = true;
    m_auditDestinationARN = std::forward<AuditDestinationARNT>(value);
  }
  template <typename AuditDestinationARNT = Aws::String>
  AssociateFileSystemRequest& WithAuditDestinationARN(AuditDestinationARNT&& value) {
    SetAuditDestinationARN(std::forward<AuditDestinationARNT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const CacheAttributes& GetCacheAttributes() const { return m_cacheAttributes; }
  inline bool CacheAttributesHasBeenSet() const { return m_cacheAttributesHasBeenSet; }
  template <typename CacheAttributesT = CacheAttributes>
  void SetCacheAttributes(CacheAttributesT&& value) {
    m_cacheAttributesHasBeenSet = true;
    m_cacheAttributes = std::forward<CacheAttributesT>(value);
  }
  template <typename CacheAttributesT = CacheAttributes>
  AssociateFileSystemRequest& WithCacheAttributes(CacheAttributesT&& value) {
    SetCacheAttributes(std::forward<CacheAttributesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the network configuration information for the gateway associated
   * with the Amazon FSx file system.</p>  <p>If multiple file systems are
   * associated with this gateway, this parameter's <code>IpAddresses</code> field is
   * required.</p>
   */
  inline const EndpointNetworkConfiguration& GetEndpointNetworkConfiguration() const { return m_endpointNetworkConfiguration; }
  inline bool EndpointNetworkConfigurationHasBeenSet() const { return m_endpointNetworkConfigurationHasBeenSet; }
  template <typename EndpointNetworkConfigurationT = EndpointNetworkConfiguration>
  void SetEndpointNetworkConfiguration(EndpointNetworkConfigurationT&& value) {
    m_endpointNetworkConfigurationHasBeenSet = true;
    m_endpointNetworkConfiguration = std::forward<EndpointNetworkConfigurationT>(value);
  }
  template <typename EndpointNetworkConfigurationT = EndpointNetworkConfiguration>
  AssociateFileSystemRequest& WithEndpointNetworkConfiguration(EndpointNetworkConfigurationT&& value) {
    SetEndpointNetworkConfiguration(std::forward<EndpointNetworkConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_userName;

  Aws::String m_password;

  Aws::String m_clientToken;

  Aws::String m_gatewayARN;

  Aws::String m_locationARN;

  Aws::Vector<Tag> m_tags;

  Aws::String m_auditDestinationARN;

  CacheAttributes m_cacheAttributes;

  EndpointNetworkConfiguration m_endpointNetworkConfiguration;
  bool m_userNameHasBeenSet = false;
  bool m_passwordHasBeenSet = false;
  bool m_clientTokenHasBeenSet = false;
  bool m_gatewayARNHasBeenSet = false;
  bool m_locationARNHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_auditDestinationARNHasBeenSet = false;
  bool m_cacheAttributesHasBeenSet = false;
  bool m_endpointNetworkConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace StorageGateway
}  // namespace Aws
