'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var React = require('react');
var ui = require('@grafana/ui');
var data = require('@grafana/data');
var runtime = require('@grafana/runtime');
var css = require('@emotion/css');
var pluginUi = require('@grafana/plugin-ui');
var lodash = require('lodash');
var asyncQueryData = require('@grafana/async-query-data');

function _interopDefaultCompat (e) { return e && typeof e === 'object' && 'default' in e ? e : { default: e }; }

var React__default = /*#__PURE__*/_interopDefaultCompat(React);

const standardRegions = [
  "af-south-1",
  "ap-east-1",
  "ap-east-2",
  "ap-northeast-1",
  "ap-northeast-2",
  "ap-northeast-3",
  "ap-south-1",
  "ap-south-2",
  "ap-southeast-1",
  "ap-southeast-2",
  "ap-southeast-3",
  "ap-southeast-4",
  "ap-southeast-5",
  "ap-southeast-6",
  "ap-southeast-7",
  "ca-central-1",
  "ca-west-1",
  "cn-north-1",
  "cn-northwest-1",
  "eu-central-1",
  "eu-central-2",
  "eu-isoe-west-1",
  "eu-north-1",
  "eu-south-1",
  "eu-south-2",
  "eu-west-1",
  "eu-west-2",
  "eu-west-3",
  "il-central-1",
  "me-central-1",
  "me-south-1",
  "mx-central-1",
  "sa-east-1",
  "us-east-1",
  "us-east-2",
  "us-gov-east-1",
  "us-gov-west-1",
  "us-iso-east-1",
  "us-iso-west-1",
  "us-isob-east-1",
  "us-isob-west-1",
  "us-isof-east-1",
  "us-northeast-1",
  "us-west-1",
  "us-west-2"
];

var AwsAuthType = /* @__PURE__ */ ((AwsAuthType2) => {
  AwsAuthType2["Keys"] = "keys";
  AwsAuthType2["Credentials"] = "credentials";
  AwsAuthType2["Default"] = "default";
  AwsAuthType2["EC2IAMRole"] = "ec2_iam_role";
  AwsAuthType2["ARN"] = "arn";
  AwsAuthType2["GrafanaAssumeRole"] = "grafana_assume_role";
  return AwsAuthType2;
})(AwsAuthType || {});

const awsAuthProviderOptions = [
  {
    label: "Workspace IAM Role",
    value: AwsAuthType.EC2IAMRole
  },
  {
    label: "Grafana Assume Role",
    value: AwsAuthType.GrafanaAssumeRole
  },
  {
    label: "AWS SDK Default",
    value: AwsAuthType.Default
  },
  {
    label: "Access & secret key",
    value: AwsAuthType.Keys
  },
  {
    label: "Credentials file",
    value: AwsAuthType.Credentials
  }
];

const assumeRoleInstructionsStyle = css.css({
  maxWidth: "715px"
});

const DEFAULT_LABEL_WIDTH = 28;
const DS_TYPES_THAT_SUPPORT_TEMP_CREDS = [
  "cloudwatch",
  "grafana-athena-datasource",
  "grafana-amazonprometheus-datasource"
];
const toOption = (value) => ({ value, label: value });
const isAwsAuthType = (value) => {
  return typeof value === "string" && awsAuthProviderOptions.some((opt) => opt.value === value);
};
const ConnectionConfig = (props) => {
  var _a, _b, _c, _d, _e, _f, _g, _h, _i;
  const [isARNInstructionsOpen, setIsARNInstructionsOpen] = React.useState(false);
  const [regions, setRegions] = React.useState((props.standardRegions || standardRegions).map(toOption));
  const {
    loadRegions,
    onOptionsChange,
    skipHeader = false,
    skipEndpoint = false,
    options,
    hideAssumeRoleArn = false
  } = props;
  let profile = options.jsonData.profile;
  if (profile === void 0) {
    profile = options.database;
  }
  const tempCredsFeatureEnabled = runtime.config.featureToggles.awsDatasourcesTempCredentials && DS_TYPES_THAT_SUPPORT_TEMP_CREDS.includes(options.type);
  const awsAssumeRoleEnabled = (_a = runtime.config.awsAssumeRoleEnabled) != null ? _a : true;
  const awsAllowedAuthProviders = React.useMemo(
    () => runtime.config.awsAllowedAuthProviders.filter((provider) => provider === AwsAuthType.GrafanaAssumeRole ? tempCredsFeatureEnabled : true).filter(isAwsAuthType),
    [tempCredsFeatureEnabled]
  );
  if (tempCredsFeatureEnabled && options.jsonData.authType === AwsAuthType.GrafanaAssumeRole) {
    if (runtime.config.namespace.startsWith("stacks-")) {
      props.externalId = runtime.config.namespace.substring(runtime.config.namespace.indexOf("-") + 1);
    }
  }
  const currentProvider = awsAuthProviderOptions.find((p) => p.value === options.jsonData.authType);
  React.useEffect(() => {
    if (!currentProvider && awsAllowedAuthProviders.length) {
      onOptionsChange({
        ...options,
        jsonData: {
          ...options.jsonData,
          authType: awsAllowedAuthProviders[0]
        }
      });
    }
  }, [currentProvider, options, onOptionsChange, awsAllowedAuthProviders]);
  React.useEffect(() => {
    if (!loadRegions) {
      return;
    }
    loadRegions().then((regions2) => setRegions(regions2.map(toOption)));
  }, [loadRegions]);
  return /* @__PURE__ */ React__default.default.createElement("div", { "data-testid": "connection-config" }, /* @__PURE__ */ React__default.default.createElement(pluginUi.ConfigSection, { title: skipHeader ? "" : "Connection Details", "data-testid": "connection-config" }, /* @__PURE__ */ React__default.default.createElement(pluginUi.ConfigSubSection, { title: "Authentication" }, /* @__PURE__ */ React__default.default.createElement(
    ui.Field,
    {
      label: "Authentication Provider",
      description: "Specify which AWS credentials chain to use.",
      htmlFor: "authProvider"
    },
    /* @__PURE__ */ React__default.default.createElement(
      ui.Select,
      {
        "aria-label": "Authentication Provider",
        inputId: "authProvider",
        value: currentProvider,
        options: awsAuthProviderOptions.filter((opt) => awsAllowedAuthProviders.includes(opt.value)),
        defaultValue: options.jsonData.authType,
        onChange: (option) => {
          data.onUpdateDatasourceJsonDataOptionSelect(props, "authType")(option);
        },
        menuShouldPortal: true
      }
    )
  ), options.jsonData.authType === "credentials" && /* @__PURE__ */ React__default.default.createElement(
    ui.Field,
    {
      label: "Credentials Profile Name",
      description: "Credentials profile name, as specified in ~/.aws/credentials, leave blank for default.",
      htmlFor: "credentialsProfileName"
    },
    /* @__PURE__ */ React__default.default.createElement(
      ui.Input,
      {
        id: "credentialsProfileName",
        placeholder: "default",
        value: options.jsonData.profile,
        onChange: data.onUpdateDatasourceJsonDataOption(props, "profile")
      }
    )
  ), options.jsonData.authType === "keys" && /* @__PURE__ */ React__default.default.createElement(React__default.default.Fragment, null, /* @__PURE__ */ React__default.default.createElement(ui.Field, { label: "Access Key ID", htmlFor: "accessKeyId" }, ((_b = props.options.secureJsonFields) == null ? void 0 : _b.accessKey) ? /* @__PURE__ */ React__default.default.createElement(ui.ButtonGroup, null, /* @__PURE__ */ React__default.default.createElement(ui.Input, { disabled: true, placeholder: "Configured", id: "accessKeyId" }), /* @__PURE__ */ React__default.default.createElement(
    ui.ToolbarButton,
    {
      icon: "edit",
      tooltip: "Edit Access Key ID",
      type: "button",
      onClick: data.onUpdateDatasourceResetOption(props, "accessKey")
    }
  )) : /* @__PURE__ */ React__default.default.createElement(
    ui.Input,
    {
      id: "accessKeyId",
      value: (_d = (_c = options.secureJsonData) == null ? void 0 : _c.accessKey) != null ? _d : "",
      onChange: data.onUpdateDatasourceSecureJsonDataOption(props, "accessKey")
    }
  )), /* @__PURE__ */ React__default.default.createElement(ui.Field, { label: "Secret Access Key", htmlFor: "secretKey" }, ((_e = props.options.secureJsonFields) == null ? void 0 : _e.secretKey) ? /* @__PURE__ */ React__default.default.createElement(ui.ButtonGroup, null, /* @__PURE__ */ React__default.default.createElement(ui.Input, { disabled: true, placeholder: "Configured" }), /* @__PURE__ */ React__default.default.createElement(
    ui.ToolbarButton,
    {
      id: "secretKey",
      icon: "edit",
      type: "button",
      tooltip: "Edit Secret Access Key",
      onClick: data.onUpdateDatasourceResetOption(props, "secretKey")
    }
  )) : /* @__PURE__ */ React__default.default.createElement(
    ui.Input,
    {
      id: "secretKey",
      value: (_g = (_f = options.secureJsonData) == null ? void 0 : _f.secretKey) != null ? _g : "",
      onChange: data.onUpdateDatasourceSecureJsonDataOption(props, "secretKey")
    }
  )))), !hideAssumeRoleArn && /* @__PURE__ */ React__default.default.createElement(
    pluginUi.ConfigSubSection,
    {
      title: "Assume Role",
      description: options.jsonData.authType === AwsAuthType.GrafanaAssumeRole ? /* @__PURE__ */ React__default.default.createElement(ui.Text, null, "Learn more about", " ", /* @__PURE__ */ React__default.default.createElement(
        ui.TextLink,
        {
          inline: true,
          external: true,
          variant: "bodySmall",
          href: "https://grafana.com/docs/plugins/cloudwatch/latest/aws-authentication/#use-grafana-assume-role"
        },
        "Grafana Assume Role"
      ), ".") : null
    },
    options.jsonData.authType === AwsAuthType.GrafanaAssumeRole && /* @__PURE__ */ React__default.default.createElement("div", { className: assumeRoleInstructionsStyle }, /* @__PURE__ */ React__default.default.createElement(
      ui.Collapse,
      {
        label: "How to create an IAM role for grafana to assume:",
        collapsible: true,
        isOpen: isARNInstructionsOpen,
        onToggle: () => setIsARNInstructionsOpen(!isARNInstructionsOpen)
      },
      /* @__PURE__ */ React__default.default.createElement("ol", null, /* @__PURE__ */ React__default.default.createElement("li", null, /* @__PURE__ */ React__default.default.createElement("p", null, "1. Create a new IAM role in the AWS console, and select ", /* @__PURE__ */ React__default.default.createElement("code", null, "Another AWS account"), " as the", " ", /* @__PURE__ */ React__default.default.createElement("code", null, "Trusted entity"), ".")), /* @__PURE__ */ React__default.default.createElement("li", null, /* @__PURE__ */ React__default.default.createElement("p", null, "2. Enter the account ID of the Grafana account that has permission to assume this role:", /* @__PURE__ */ React__default.default.createElement("code", null, " 008923505280 "), " and check the ", /* @__PURE__ */ React__default.default.createElement("code", null, "Require external ID"), " box.")), /* @__PURE__ */ React__default.default.createElement("li", null, /* @__PURE__ */ React__default.default.createElement("p", null, "3. Enter the following external ID:", " ", /* @__PURE__ */ React__default.default.createElement("code", null, props.externalId || "External Id is currently unavailable"), " and click", " ", /* @__PURE__ */ React__default.default.createElement("code", null, "Next"), ".")), /* @__PURE__ */ React__default.default.createElement("li", null, /* @__PURE__ */ React__default.default.createElement("p", null, "4. Add any required permissions you would like Grafana to be able to access on your behalf. For more details on our permissions please", " ", /* @__PURE__ */ React__default.default.createElement(
        "a",
        {
          href: "https://grafana.com/docs/grafana/latest/datasources/aws-cloudwatch/",
          target: "_blank",
          rel: "noreferrer"
        },
        "read through our documentation"
      ), ".")), /* @__PURE__ */ React__default.default.createElement("li", null, /* @__PURE__ */ React__default.default.createElement("p", null, "5. Give the role a name and description, and click ", /* @__PURE__ */ React__default.default.createElement("code", null, "Create role"), ".")), /* @__PURE__ */ React__default.default.createElement("li", null, /* @__PURE__ */ React__default.default.createElement("p", null, "6. Copy the ARN of the role you just created and paste it into the ", /* @__PURE__ */ React__default.default.createElement("code", null, "Assume Role ARN"), " ", "field below.")))
    )),
    awsAssumeRoleEnabled && /* @__PURE__ */ React__default.default.createElement(React__default.default.Fragment, null, /* @__PURE__ */ React__default.default.createElement(
      ui.Field,
      {
        htmlFor: "assumeRoleArn",
        label: "Assume Role ARN",
        description: "Optional. Specifying the ARN of a role will ensure that the\n                     selected authentication provider is used to assume the role rather than the\n                     credentials directly."
      },
      /* @__PURE__ */ React__default.default.createElement(
        ui.Input,
        {
          id: "assumeRoleArn",
          placeholder: "arn:aws:iam:*",
          value: options.jsonData.assumeRoleArn || "",
          onChange: data.onUpdateDatasourceJsonDataOption(props, "assumeRoleArn")
        }
      )
    ), options.jsonData.authType !== AwsAuthType.GrafanaAssumeRole && /* @__PURE__ */ React__default.default.createElement(
      ui.Field,
      {
        htmlFor: "externalId",
        label: "External ID",
        description: "If you are assuming a role in another account, that has been created with an external ID, specify the external ID here."
      },
      /* @__PURE__ */ React__default.default.createElement(
        ui.Input,
        {
          id: "externalId",
          placeholder: "External ID",
          value: options.jsonData.externalId || "",
          onChange: data.onUpdateDatasourceJsonDataOption(props, "externalId")
        }
      )
    ))
  ), /* @__PURE__ */ React__default.default.createElement(pluginUi.ConfigSubSection, { title: "Additional Settings" }, !skipEndpoint && options.jsonData.authType !== AwsAuthType.GrafanaAssumeRole && /* @__PURE__ */ React__default.default.createElement(
    ui.Field,
    {
      label: "Endpoint",
      description: "Optionally, specify a custom endpoint for the service",
      htmlFor: "endpoint"
    },
    /* @__PURE__ */ React__default.default.createElement(
      ui.Input,
      {
        id: "endpoint",
        placeholder: (_h = props.defaultEndpoint) != null ? _h : "https://{service}.{region}.amazonaws.com",
        value: options.jsonData.endpoint || "",
        onChange: data.onUpdateDatasourceJsonDataOption(props, "endpoint")
      }
    )
  ), /* @__PURE__ */ React__default.default.createElement(
    ui.Field,
    {
      label: "Default Region",
      description: "Specify the region, such as for US West (Oregon) use ` us-west-2 ` as the region.",
      htmlFor: "defaultRegion"
    },
    /* @__PURE__ */ React__default.default.createElement(
      ui.Select,
      {
        inputId: "defaultRegion",
        value: (_i = regions.find((region) => region.value === options.jsonData.defaultRegion)) != null ? _i : options.jsonData.defaultRegion ? {
          label: options.jsonData.defaultRegion,
          value: options.jsonData.defaultRegion
        } : void 0,
        options: regions,
        defaultValue: options.jsonData.defaultRegion,
        allowCustomValue: true,
        onChange: data.onUpdateDatasourceJsonDataOptionSelect(props, "defaultRegion"),
        formatCreateLabel: (r) => `Use region: ${r}`,
        menuShouldPortal: true
      }
    )
  )), props.children));
};

const versionPattern = /^(\d+)(?:\.(\d+))?(?:\.(\d+))?(?:-([0-9A-Za-z\.]+))?/;
class SemVersion {
  constructor(version) {
    this.major = 0;
    this.minor = 0;
    this.patch = 0;
    this.meta = "";
    const match = versionPattern.exec(version);
    if (match) {
      this.major = Number(match[1]);
      this.minor = Number(match[2] || 0);
      this.patch = Number(match[3] || 0);
      this.meta = match[4];
    }
  }
  isGtOrEq(version) {
    const compared = new SemVersion(version);
    for (let i = 0; i < this.comparable.length; ++i) {
      if (this.comparable[i] > compared.comparable[i]) {
        return true;
      }
      if (this.comparable[i] < compared.comparable[i]) {
        return false;
      }
    }
    return true;
  }
  isValid() {
    return lodash.isNumber(this.major);
  }
  get comparable() {
    return [this.major, this.minor, this.patch];
  }
}
function isVersionGtOrEq(a, b) {
  const aSemver = new SemVersion(a);
  return aSemver.isGtOrEq(b);
}

function Divider() {
  const theme = ui.useTheme2();
  if (isVersionGtOrEq(runtime.config.buildInfo.version, "10.1.0")) {
    return /* @__PURE__ */ React__default.default.createElement(ui.Divider, null);
  }
  return /* @__PURE__ */ React__default.default.createElement(
    "div",
    {
      style: { borderTop: `1px solid ${theme.colors.border.weak}`, margin: theme.spacing(2, 0), width: "100%" }
    }
  );
}

const SIGV4ConnectionConfig = (props) => {
  var _a, _b, _c, _d;
  const { onOptionsChange, options } = props;
  const connectionConfigProps = {
    onOptionsChange: (awsDataSourceSettings) => {
      var _a2, _b2, _c2, _d2;
      const dataSourceSettings = {
        ...options,
        jsonData: {
          ...options.jsonData,
          sigV4AuthType: awsDataSourceSettings.jsonData.authType,
          sigV4Profile: awsDataSourceSettings.jsonData.profile,
          sigV4AssumeRoleArn: awsDataSourceSettings.jsonData.assumeRoleArn,
          sigV4ExternalId: awsDataSourceSettings.jsonData.externalId,
          sigV4Region: awsDataSourceSettings.jsonData.defaultRegion,
          sigV4Endpoint: awsDataSourceSettings.jsonData.endpoint
        },
        secureJsonFields: {
          sigV4AccessKey: (_a2 = awsDataSourceSettings.secureJsonFields) == null ? void 0 : _a2.accessKey,
          sigV4SecretKey: (_b2 = awsDataSourceSettings.secureJsonFields) == null ? void 0 : _b2.secretKey
        },
        secureJsonData: {
          sigV4AccessKey: (_c2 = awsDataSourceSettings.secureJsonData) == null ? void 0 : _c2.accessKey,
          sigV4SecretKey: (_d2 = awsDataSourceSettings.secureJsonData) == null ? void 0 : _d2.secretKey
        }
      };
      onOptionsChange(dataSourceSettings);
    },
    options: {
      ...options,
      jsonData: {
        ...options.jsonData,
        authType: options.jsonData.sigV4AuthType,
        profile: options.jsonData.sigV4Profile,
        assumeRoleArn: options.jsonData.sigV4AssumeRoleArn,
        externalId: options.jsonData.sigV4ExternalId,
        defaultRegion: options.jsonData.sigV4Region,
        endpoint: options.jsonData.sigV4Endpoint
      },
      secureJsonFields: {
        accessKey: (_a = options.secureJsonFields) == null ? void 0 : _a.sigV4AccessKey,
        secretKey: (_b = options.secureJsonFields) == null ? void 0 : _b.sigV4SecretKey
      },
      secureJsonData: {
        accessKey: (_c = options.secureJsonData) == null ? void 0 : _c.sigV4AccessKey,
        secretKey: (_d = options.secureJsonData) == null ? void 0 : _d.sigV4SecretKey
      }
    },
    inExperimentalAuthComponent: props.inExperimentalAuthComponent
  };
  return /* @__PURE__ */ React__default.default.createElement(React__default.default.Fragment, null, /* @__PURE__ */ React__default.default.createElement("div", { className: "gf-form" }, /* @__PURE__ */ React__default.default.createElement("h6", null, "SigV4 Auth Details")), /* @__PURE__ */ React__default.default.createElement(ConnectionConfig, { ...connectionConfigProps, skipHeader: true, skipEndpoint: true }));
};

const defaultKey = "__default";

function ResourceSelector(props) {
  const propsDependencies = props.dependencies;
  const propsOnChange = props.onChange;
  const dependencies = React.useRef(props.dependencies);
  const fetched = React.useRef(false);
  const resource = React.useRef(props.value || props.default || null);
  const [resources, setResources] = React.useState(
    resource.current ? [resource.current] : []
  );
  const [isLoading, setIsLoading] = React.useState(false);
  const defaultOpts = React.useMemo(() => {
    const opts = [
      {
        label: `default (${props.default})`,
        value: defaultKey,
        description: `Default value set in the data source`
      }
    ];
    if (props.value && props.value !== defaultKey) {
      opts.push({ label: props.value, value: props.value });
    }
    return opts;
  }, [props.default, props.value]);
  const [options, setOptions] = React.useState(props.default ? defaultOpts : []);
  React.useEffect(() => {
    if (props.resources !== void 0) {
      setResources(props.resources);
    }
  }, [props.resources]);
  React.useEffect(() => {
    const newOptions = props.default ? defaultOpts : [];
    if (resources.length) {
      resources.forEach((r) => {
        const value = typeof r === "string" ? r : r.value;
        if (!newOptions.find((o) => o.value === value)) {
          typeof r === "string" ? newOptions.push({ label: r, value: r }) : newOptions.push(r);
        }
      });
      setOptions(newOptions);
    } else {
      setOptions([]);
    }
  }, [resources, defaultOpts, props.default]);
  React.useEffect(() => {
    if (!lodash.isEqual(propsDependencies, dependencies.current)) {
      fetched.current = false;
      resource.current = null;
      dependencies.current = propsDependencies;
      propsOnChange(null);
    }
  }, [propsDependencies, propsOnChange]);
  const fetch = async () => {
    var _a;
    if (fetched.current) {
      return;
    }
    if (props.saveOptions) {
      await props.saveOptions();
    }
    try {
      const resources2 = await ((_a = props.fetch) == null ? void 0 : _a.call(props)) || [];
      setResources(resources2);
    } finally {
      fetched.current = true;
    }
  };
  const onChange = (e) => {
    propsOnChange(e);
    if (e.value) {
      resource.current = e.value;
    }
  };
  const onClick = async () => {
    setIsLoading(true);
    try {
      await fetch();
    } finally {
      setIsLoading(false);
    }
  };
  return /* @__PURE__ */ React__default.default.createElement(
    ui.Select,
    {
      ...props,
      id: props.id,
      inputId: props.id,
      "aria-label": props.label,
      options,
      onChange,
      isLoading,
      className: props.className || "min-width-6",
      onOpenMenu: () => props.fetch && onClick(),
      menuShouldPortal: true
    }
  );
}

function ConfigSelect(props) {
  var _a, _b, _c;
  const { jsonData } = props.options;
  const commonProps = {
    title: jsonData.defaultRegion ? "" : "select a default region",
    labelWidth: (_a = props.labelWidth) != null ? _a : DEFAULT_LABEL_WIDTH,
    className: "width-30"
  };
  const dependencies = [
    props.options.jsonData.assumeRoleArn,
    props.options.jsonData.authType,
    props.options.jsonData.defaultRegion,
    props.options.jsonData.endpoint,
    props.options.jsonData.externalId,
    props.options.jsonData.profile,
    (_b = props.options.secureJsonData) == null ? void 0 : _b.accessKey,
    (_c = props.options.secureJsonData) == null ? void 0 : _c.secretKey
  ].concat(props.dependencies);
  return /* @__PURE__ */ React__default.default.createElement(
    ResourceSelector,
    {
      id: props.id,
      label: props.label,
      "data-testid": props["data-testid"],
      onChange: props.onChange,
      fetch: props.fetch,
      value: props.value,
      saveOptions: props.saveOptions,
      dependencies,
      hidden: props.hidden,
      disabled: props.disabled || !jsonData.defaultRegion,
      allowCustomValue: props.allowCustomValue,
      autoFocus: props.autoFocus,
      backspaceRemovesValue: props.backspaceRemovesValue,
      invalid: props.invalid,
      isClearable: props.isClearable,
      isMulti: props.isMulti,
      inputId: props.inputId,
      showAllSelectedWhenOpen: props.showAllSelectedWhenOpen,
      maxMenuHeight: props.maxMenuHeight,
      minMenuHeight: props.minMenuHeight,
      maxVisibleValues: props.maxVisibleValues,
      menuPlacement: props.menuPlacement,
      menuPosition: props.menuPosition,
      noOptionsMessage: props.noOptionsMessage,
      placeholder: props.placeholder,
      width: props.width,
      onBlur: props.onBlur,
      onCreateOption: props.onCreateOption,
      onInputChange: props.onInputChange,
      isOptionDisabled: props.isOptionDisabled,
      ...commonProps
    }
  );
}

function InlineInput(props) {
  var _a;
  return /* @__PURE__ */ React__default.default.createElement(
    ui.InlineField,
    {
      label: props.label,
      labelWidth: (_a = props.labelWidth) != null ? _a : DEFAULT_LABEL_WIDTH,
      tooltip: props.tooltip,
      hidden: props.hidden,
      disabled: props.disabled
    },
    /* @__PURE__ */ React__default.default.createElement(
      ui.Input,
      {
        "data-testid": props["data-testid"],
        className: "width-30",
        value: props.value,
        onChange: props.onChange,
        placeholder: props.placeholder,
        disabled: props.disabled
      }
    )
  );
}

function QueryCodeEditor(props) {
  const { getSuggestions, query } = props;
  const { rawSQL } = lodash.defaults(props.query, { rawSQL: "" });
  const onRawSqlChange = (rawSQL2) => {
    const query2 = {
      ...props.query,
      rawSQL: rawSQL2
    };
    props.onChange(query2);
    props.onRunQuery();
  };
  const suggestionsRef = React.useRef([]);
  React.useEffect(() => {
    suggestionsRef.current = getSuggestions(query);
  }, [getSuggestions, query]);
  return /* @__PURE__ */ React__default.default.createElement(
    ui.CodeEditor,
    {
      language: props.language,
      value: rawSQL,
      onBlur: onRawSqlChange,
      showMiniMap: false,
      showLineNumbers: true,
      getSuggestions: () => suggestionsRef.current,
      height: "240px",
      ...props.editorProps
    }
  );
}

function QueryEditorHeader({
  query,
  showAsyncQueryButtons,
  extraHeaderElementLeft,
  extraHeaderElementRight,
  enableRunButton,
  onRunQuery,
  data: data$1,
  cancel
}) {
  return /* @__PURE__ */ React__default.default.createElement(pluginUi.EditorHeader, null, extraHeaderElementLeft, /* @__PURE__ */ React__default.default.createElement(pluginUi.FlexItem, { grow: 1 }), showAsyncQueryButtons ? /* @__PURE__ */ React__default.default.createElement(
    asyncQueryData.RunQueryButtons,
    {
      onRunQuery,
      enableRun: enableRunButton,
      query,
      onCancelQuery: (target) => {
        cancel == null ? void 0 : cancel(target);
      },
      state: data$1 == null ? void 0 : data$1.state
    }
  ) : /* @__PURE__ */ React__default.default.createElement(
    ui.Button,
    {
      variant: enableRunButton ? "primary" : "secondary",
      size: "sm",
      onClick: onRunQuery,
      icon: (data$1 == null ? void 0 : data$1.state) === data.LoadingState.Loading ? "fa fa-spinner" : void 0,
      disabled: (data$1 == null ? void 0 : data$1.state) === data.LoadingState.Loading || !enableRunButton
    },
    "Run queries"
  ), extraHeaderElementRight);
}

function FormatSelect(props) {
  var _a;
  const onChangeFormat = (e) => {
    var _a2;
    props.onChange({
      ...props.query,
      format: e.value || 0
    });
    (_a2 = props.onRunQuery) == null ? void 0 : _a2.call(props);
  };
  return /* @__PURE__ */ React__default.default.createElement(
    ui.Select,
    {
      "aria-label": "Format data frames as",
      id: (_a = props.id) != null ? _a : "formatAs",
      options: props.options,
      value: props.query.format,
      onChange: onChangeFormat,
      menuShouldPortal: true
    }
  );
}

var FillValueOptions = /* @__PURE__ */ ((FillValueOptions2) => {
  FillValueOptions2[FillValueOptions2["Previous"] = 0] = "Previous";
  FillValueOptions2[FillValueOptions2["Null"] = 1] = "Null";
  FillValueOptions2[FillValueOptions2["Value"] = 2] = "Value";
  return FillValueOptions2;
})(FillValueOptions || {});
const SelectableFillValueOptions = [
  {
    label: "Previous Value",
    value: 0 /* Previous */
  },
  {
    label: "NULL",
    value: 1 /* Null */
  },
  {
    label: "Value",
    value: 2 /* Value */
  }
];
function FillValueSelect(props) {
  var _a, _b, _c;
  return /* @__PURE__ */ React__default.default.createElement(React__default.default.Fragment, null, /* @__PURE__ */ React__default.default.createElement(pluginUi.EditorField, { label: "Fill with", tooltip: "value to fill missing points", htmlFor: "fillWith" }, /* @__PURE__ */ React__default.default.createElement(
    ui.Select,
    {
      id: "fillWith",
      "aria-label": "Fill with",
      "data-testid": "table-fill-with-select",
      options: SelectableFillValueOptions,
      value: (_b = (_a = props.query.fillMode) == null ? void 0 : _a.mode) != null ? _b : 0 /* Previous */,
      onChange: ({ value }) => {
        var _a2;
        props.onChange({
          ...props.query,
          // Keep the fillMode.value in case FillValueOptions.Value mode is selected back
          fillMode: { ...props.query.fillMode, mode: value }
        });
        (_a2 = props.onRunQuery) == null ? void 0 : _a2.call(props);
      },
      menuShouldPortal: true
    }
  )), ((_c = props.query.fillMode) == null ? void 0 : _c.mode) === 2 /* Value */ && /* @__PURE__ */ React__default.default.createElement(pluginUi.EditorField, { label: "Value", htmlFor: "valueToFill", width: 6 }, /* @__PURE__ */ React__default.default.createElement(
    ui.Input,
    {
      id: "valueToFill",
      "aria-label": "Value",
      type: "number",
      value: props.query.fillMode.value,
      onChange: ({ currentTarget }) => props.onChange({
        ...props.query,
        fillMode: {
          mode: 2 /* Value */,
          value: currentTarget.valueAsNumber
        }
      }),
      onBlur: () => {
        var _a2;
        return (_a2 = props.onRunQuery) == null ? void 0 : _a2.call(props);
      }
    }
  )));
}

function filterSQLQuery(query) {
  return !!query.rawSQL;
}
function applySQLTemplateVariables(query, scopedVars, getTemplateSrv) {
  const templateSrv = getTemplateSrv();
  return {
    ...query,
    rawSQL: templateSrv.replace(query.rawSQL, scopedVars, interpolateVariable)
  };
}
function interpolateVariable(value) {
  if (typeof value === "string" || typeof value === "number") {
    return value;
  }
  const quotedValues = value.map((v) => {
    return quoteLiteral(v);
  });
  return quotedValues.join(",");
}
function quoteLiteral(value) {
  return "'" + String(value).replace(/'/g, "''") + "'";
}
const appendTemplateVariablesAsSuggestions = (getTemplateSrv, sugs) => {
  const templateSrv = getTemplateSrv();
  const templateSugs = [];
  templateSrv.getVariables().forEach((variable) => {
    const label = "$" + variable.name;
    let val = templateSrv.replace(label);
    if (val === label) {
      val = "";
    }
    templateSugs.push({
      label,
      kind: ui.CodeEditorSuggestionItemKind.Text,
      detail: `(Template Variable) ${val}`
    });
  });
  return sugs.concat(templateSugs);
};

exports.AwsAuthType = AwsAuthType;
exports.ConfigSelect = ConfigSelect;
exports.ConnectionConfig = ConnectionConfig;
exports.DEFAULT_LABEL_WIDTH = DEFAULT_LABEL_WIDTH;
exports.Divider = Divider;
exports.FillValueOptions = FillValueOptions;
exports.FillValueSelect = FillValueSelect;
exports.FormatSelect = FormatSelect;
exports.InlineInput = InlineInput;
exports.QueryCodeEditor = QueryCodeEditor;
exports.QueryEditorHeader = QueryEditorHeader;
exports.ResourceSelector = ResourceSelector;
exports.SIGV4ConnectionConfig = SIGV4ConnectionConfig;
exports.appendTemplateVariablesAsSuggestions = appendTemplateVariablesAsSuggestions;
exports.applySQLTemplateVariables = applySQLTemplateVariables;
exports.awsAuthProviderOptions = awsAuthProviderOptions;
exports.filterSQLQuery = filterSQLQuery;
exports.standardRegions = standardRegions;
//# sourceMappingURL=index.js.map
