// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { isMultiRegionAwsKmsIdentifier, parseAwsKmsKeyArn } from './arn_parsing';
//= compliance/framework/aws-kms/aws-kms-mrk-are-unique.txt#2.5
//# The caller MUST provide:
export function awsKmsMrkAreUnique(awsKmsIdentifers) {
    const multiRegionKeys = awsKmsIdentifers.filter((i) => isMultiRegionAwsKmsIdentifier(i));
    //= compliance/framework/aws-kms/aws-kms-mrk-are-unique.txt#2.5
    //# If the list does not contain any multi-Region keys (aws-kms-key-
    //# arn.md#identifying-an-aws-kms-multi-region-key) this function MUST
    //# exit successfully.
    if (!multiRegionKeys.length)
        return;
    const multiRegionKeyIds = multiRegionKeys.map((mrk) => {
        const arn = parseAwsKmsKeyArn(mrk);
        return arn ? arn.ResourceId : mrk;
    });
    //= compliance/framework/aws-kms/aws-kms-mrk-are-unique.txt#2.5
    //# If there are zero duplicate resource ids between the multi-region
    //# keys, this function MUST exit successfully
    if (new Set(multiRegionKeyIds).size === multiRegionKeys.length)
        return;
    //= compliance/framework/aws-kms/aws-kms-mrk-are-unique.txt#2.5
    //# If any duplicate multi-region resource ids exist, this function MUST
    //# yield an error that includes all identifiers with duplicate resource
    //# ids not only the first duplicate found.
    const duplicateMultiRegionIdentifiers = multiRegionKeyIds
        .map((mrk, i, a) => {
        if (a.indexOf(mrk) !== a.lastIndexOf(mrk))
            return multiRegionKeys[i];
        /* Postcondition: Remove non-duplicate multi-Region keys. */
        return false;
    })
        .filter((dup) => dup)
        .join(',');
    throw new Error(`Related multi-Region keys: ${duplicateMultiRegionIdentifiers} are not allowed.`);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzX2ttc19tcmtfYXJlX3VuaXF1ZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9hd3Nfa21zX21ya19hcmVfdW5pcXVlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7QUFFdEMsT0FBTyxFQUFFLDZCQUE2QixFQUFFLGlCQUFpQixFQUFFLE1BQU0sZUFBZSxDQUFBO0FBRWhGLCtEQUErRDtBQUMvRCw0QkFBNEI7QUFDNUIsTUFBTSxVQUFVLGtCQUFrQixDQUFDLGdCQUEwQjtJQUMzRCxNQUFNLGVBQWUsR0FBRyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUNwRCw2QkFBNkIsQ0FBQyxDQUFDLENBQUMsQ0FDakMsQ0FBQTtJQUVELCtEQUErRDtJQUMvRCxvRUFBb0U7SUFDcEUsc0VBQXNFO0lBQ3RFLHNCQUFzQjtJQUN0QixJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU07UUFBRSxPQUFNO0lBRW5DLE1BQU0saUJBQWlCLEdBQUcsZUFBZSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFO1FBQ3BELE1BQU0sR0FBRyxHQUFHLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxDQUFBO1FBQ2xDLE9BQU8sR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUE7SUFDbkMsQ0FBQyxDQUFDLENBQUE7SUFDRiwrREFBK0Q7SUFDL0QscUVBQXFFO0lBQ3JFLDhDQUE4QztJQUM5QyxJQUFJLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLENBQUMsSUFBSSxLQUFLLGVBQWUsQ0FBQyxNQUFNO1FBQUUsT0FBTTtJQUV0RSwrREFBK0Q7SUFDL0Qsd0VBQXdFO0lBQ3hFLHdFQUF3RTtJQUN4RSwyQ0FBMkM7SUFDM0MsTUFBTSwrQkFBK0IsR0FBRyxpQkFBaUI7U0FDdEQsR0FBRyxDQUFDLENBQUMsR0FBRyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRTtRQUNqQixJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUM7WUFBRSxPQUFPLGVBQWUsQ0FBQyxDQUFDLENBQUMsQ0FBQTtRQUNwRSw0REFBNEQ7UUFDNUQsT0FBTyxLQUFLLENBQUE7SUFDZCxDQUFDLENBQUM7U0FDRCxNQUFNLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLENBQUE7SUFFWixNQUFNLElBQUksS0FBSyxDQUNiLDhCQUE4QiwrQkFBK0IsbUJBQW1CLENBQ2pGLENBQUE7QUFDSCxDQUFDIn0=